/*  Ermittelt die zu verwendente Belastung der vorgegebenen Kopfkostenstelle an dem vorgegeben Tag.

    Über das Setting "scheduling.algorithm_is_kapa_stack" erfolgt die Wahl zwischen den zwei Konzepten:
    "Kapazitätsstack"
    oder
    "Maximale Parallelität".

    Kapazitätsstack:
    Die Kapazität der Kopfkostenstelle (KKS) wird einfach als Summe von Arbeitsstunden aufgefasst, welche die KKS innerhalb
    eines Zeitraums leisten kann. Jeder auf KKS eingeplante Belegungsslot legt eine neue Belastung auf den "Stack" (Stapel) der
    KKS. Die Dicke ti_usage entspricht der hinzugekommen verbrauchten Kapazität. Der Belegungsslot auf der KKS wird den ganzen
    Tag eingeplant. Wird der optionale Parameter _factor_tm_ta angegeben, wird mit der anteiligen Personalzeit anstatt mit der
    Abarbeitungszeit als Belegung gerechnet.

    Maximale Parallelität:
    Die Idee: Die Tageskapazität ist größer als die maximale Tagesarbeitszeit auf der Kopfkostenstelle.
    Der Faktor gibt an, wieviele parallele Arbeiten die Kopfkostenstelle bei maximaler Auslastung maximal durchführen kann.
    Die einzuplanende Belastung ist reziprok zum obigen Faktor und ist daher entsprechend kleiner, je größer der Faktor.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__timeslots_top_usage__calc', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__timeslots_top_usage__calc(
    _top_resource_id  integer,
    _date  date,
    _slotSizeUsed numeric,
    _factor_tm_ta numeric = 1.0,        -- Personalzeit-zu-Abarbeitungszeit-Faktor = a2_tm / a2_ta
    _loglevel int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS numeric AS $$

  DECLARE

      _prefix               varchar := format( 'resource_timeline__timeslots_top_usage__calc rid %L -', _top_resource_id );
      _algorithm            varchar := 'max parallelism';

      _top_usage            numeric(12,8);
      _top_ksvrecord        record;
      _dayofweek            integer;
      _max_worktime_day     numeric;
      _top_kapa_day         numeric;
      _tolerated_overload   constant numeric := TSystem.Settings__GetInteger( 'scheduling.tolerated_overload', 25 );  -- zugelassende Überlastung der KKS in %.

  BEGIN
      -- Debug
      IF _loglevel >= 5 THEN
          IF TSystem.Settings__GetBool( 'scheduling.algorithm_is_kapa_stack' ) THEN
              _algorithm := 'kapacity stack';
          END IF;
          RAISE NOTICE '%', format( 'call: scheduling.resource_timeline__timeslots_top_usage__calc( _top_resource_id => %L, _date => %L, _slotSizeUsed => %L, _factor_tm_ta => %L, _loglevel => %L ) algorithm: %L',
                                                                                                    _top_resource_id      , _date      , round( _slotSizeUsed, 4 ), round( _factor_tm_ta, 4 ), _loglevel, _algorithm
                            )
          ;
      END IF;

      -- Es wurde keine Kopfkostenstelle gefunden, dann kann es auch keine Belastung geben.
      IF _top_resource_id IS null THEN
          _top_usage := null;
          RETURN _top_usage;
      END IF;

      -- Die Kopfkostenstelle aus der resourcen_id ermitteln.
      SELECT ksv.*
        INTO _top_ksvrecord
        FROM ksv
        JOIN ksvba ON ksb_ks_id = ks_id
        JOIN scheduling.resource ON context_id = ksb_id
                                AND context = 'ksvba'
       WHERE resource.id = _top_resource_id;

      -- Maximale Arbeitszeit der Kostenstelle am vorgegebenen Tag ermitteln.
      IF NOT TSystem.Settings__GetBool( 'scheduling.algorithm_is_kapa_stack' ) THEN
          _dayofweek := day_of_week( _date );
          IF    _dayofweek = 0 THEN _max_worktime_day := _top_ksvrecord.ks_ka1;--sonntag
          ELSIF _dayofweek = 1 THEN _max_worktime_day := _top_ksvrecord.ks_ka2;--montag
          ELSIF _dayofweek = 2 THEN _max_worktime_day := _top_ksvrecord.ks_ka3;--dienstag
          ELSIF _dayofweek = 3 THEN _max_worktime_day := _top_ksvrecord.ks_ka4;--mittwoch
          ELSIF _dayofweek = 4 THEN _max_worktime_day := _top_ksvrecord.ks_ka5;--donnerstag
          ELSIF _dayofweek = 5 THEN _max_worktime_day := _top_ksvrecord.ks_ka6;--freitag
          ELSIF _dayofweek = 6 THEN _max_worktime_day := _top_ksvrecord.ks_ka7;--samstag
          END IF;
      END IF;

      -- Die Kapazität der Kostenstelle am vorgegebenen Tag ermitteln.
      _top_kapa_day := tplanterm.ks_day_kapa( _top_resource_id, _date );

      -- Berechnung der zu verwendenen Belastung der Kostenstelle.
      IF TSystem.Settings__GetBool( 'scheduling.algorithm_is_kapa_stack' ) THEN
          -- Wenn der Faktor 0 oder null ist (a2_tm ist nicht gesetzt), dann 1 annehmen. Also Personalzeit entspricht Abarbeitungszeit.
          IF ( _factor_tm_ta = 0.0 OR _factor_tm_ta IS NULL ) THEN
              _factor_tm_ta := 1.0;
          END IF;
          -- Es ist  ein Arbeitsfreier Tag ...
          IF _top_kapa_day = 0 THEN
              -- ... dann Belastung der KKS als 1 annehmen (Division durch 0 verhindern).
              -- Debug
              IF _loglevel >= 5 THEN
                  RAISE NOTICE '% _top_kapa_day = 0', _prefix;
              END IF;
              _top_usage := 1;
          -- Es ist KEIN arbeitsfreier Tag ...
          ELSE
              -- ... dann Belastung der KKS mit folgender Formel ausrechnen.
              -- Debug Formel
              IF _loglevel >= 5 THEN
                  RAISE NOTICE '% ( % * % ) / ( % * % )', _prefix, round( _slotSizeUsed, 4 ), round( _factor_tm_ta, 4 ), round( ( 1 + ( _tolerated_overload / 100 ) ), 4 ), round( nullif( _top_kapa_day, 1 ), 4 );
              END IF;
              -- Belastung = ( ( _slotSizeUsed * _factor_tm_ta ) / ( 1,25 * _top_kapa_day ) = anteilige Personalzeit / ( ( 1 + erlaubte Überlast) * Tageskapazität der KKS )
              _top_usage := ( _slotSizeUsed * _factor_tm_ta ) / ( ( 1 + ( _tolerated_overload / 100 ) ) * nullif( _top_kapa_day, 1 ) );
          END IF;

      ELSE
          -- Debug Formel
          IF _loglevel >= 5 THEN
              RAISE NOTICE '% 1 / ceil( % * %  / % )', _prefix, round( ( 1 + ( _tolerated_overload / 100 ) ), 4 ), round( nullif( _top_kapa_day, 0 ), 4), round( _max_worktime_day, 4 );
          END IF;
          -- Belastung =  1 / 1.25 * Anzahl Arbeiter -> (25% Überlast zulassen)
          _top_usage := 1 / ceil ( ( 1 + ( _tolerated_overload / 100 ) ) * nullif( _top_kapa_day, 0 ) / _max_worktime_day );
      END IF;

      -- Fälle abfangen, ...
      -- ... in denen die maximale Tagesarbeitszeit oder die Tageskapazität 0 ist
      -- oder
      -- ... in denen maximale Tagesarbeitszeit kleiner als die die Tageskapazität ist.
      IF
            _top_usage IS null
        OR  _top_usage = 0
        OR  nullif( _max_worktime_day, 0 ) > nullif( _top_kapa_day, 0 )
      THEN
          IF TSystem.Settings__GetBool( 'scheduling.algorithm_is_kapa_stack' ) THEN
              _top_usage := 1;
          ELSE
              _top_usage := 1;
          END IF;
      END IF;

      RETURN _top_usage;

  END $$ language plpgsql stable;
--